/************************************************************************
 *
 * \file: AutoSmoketest.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 *  implements functionality to automate the smoketest.
 * <detailed description>
 * \component: Android Auto - Demo application
 *
 * \author:\author: Ajay Kumar Sahoo / RBEI/ECF3 /ajaykumar.sahoo@in.bosch.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <adit_logging.h>
#include <signal.h>

#include <uspi/FeatureDiscoveryTypes.h>
#include "AutoSmoketest.h"

LOG_IMPORT_CONTEXT(demo)

namespace adit { namespace aauto {

using namespace utility;

AutoSmoketest::AutoSmoketest()
{
    static int _counter = 0;
    _counter++;
    if (_counter > 1)
        fprintf(stderr, "AutoSmoketest object shall not be used twice");

    stTimeout = 20;
    timerid = 0;
    timerCreated = false;
    stError =  NOERROR; //default
    stMode = MANUAL;
    displayId = 0;
    mGraphicDumpDelay = 0;
    mTestResult = -1;
}

AutoSmoketest::~AutoSmoketest()
{
    deleteSignalTimer();
}

void AutoSmoketest::waitForExit()
{
    std::unique_lock<std::mutex> _lock(mMtxWaitForExit);
    mCvWaitForExit.wait(_lock);
}

void AutoSmoketest::requestStop()
{
    LOG_WARN((demo, "AutoSmoketest::%s()  ", __func__));

    std::lock_guard<std::mutex> guard(mMtxWaitForExit);
    mCvWaitForExit.notify_one();
}

void AutoSmoketest::setTestError(testError err)
{
    stError = err;
    if (stError == VIDEOFRAMERENDERED)
    {
        stError = DONE;

        std::lock_guard<std::mutex> guard(mMtxWaitForExit);
        mCvWaitForExit.notify_one();
    }
}

void AutoSmoketest::addDevice(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inDeviceToAdd)
{
    LOGD_DEBUG((demo, "%s()  add device to end of device list", __func__));
    std::unique_lock<std::mutex> _lock(mMtxDevInfos);
    usbDevInfos.push_back(inDeviceToAdd);
}

void AutoSmoketest::removeDevice(void)
{
    std::unique_lock<std::mutex> _lock(mMtxDevInfos);
    if (true != usbDevInfos.empty())
    {
        LOGD_DEBUG((demo, "%s()  remove first device from device list", __func__));
        usbDevInfos.pop_front();
    }
    else
    {
        LOG_INFO((demo, "%s()   cannot remove device because device list is empty", __func__));
    }

}

std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> AutoSmoketest::getTestDevice(void)
{
    std::unique_lock<std::mutex> _lock(mMtxDevInfos);
    if (true == usbDevInfos.empty())
    {
        LOG_INFO((demo, "%s()   device list is empty", __func__));
        return nullptr;
    }

    return usbDevInfos.front();
}

std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> AutoSmoketest::getTestDevice(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDev)
{
    std::unique_lock<std::mutex> _lock(mMtxDevInfos);

    if (true != usbDevInfos.empty())
    {
        /* go through all queued devices to find the request one */
        for(std::list<std::shared_ptr<adit::uspi::DiscoveredDeviceUsb>>::iterator iter = usbDevInfos.begin(); iter != usbDevInfos.end(); ++iter)
        {
            uspi::DeviceInfo ddInfo = iter.operator *()->getInfo();

            LOGD_DEBUG((demo, "%s()  device in list: serial=%s",
                                __func__, ddInfo.getDeviceInfo(uspi::DSYS_SERIAL).c_str() ));

            if ( 0 == inUsbDev->getInfo().getDeviceInfo(uspi::DSYS_SERIAL).compare(ddInfo.getDeviceInfo(uspi::DSYS_SERIAL).c_str()) )
            {
                LOGD_DEBUG((demo, "%s()  found device with serial=%s",
                        __func__, ddInfo.getDeviceInfo(uspi::DSYS_SERIAL).c_str() ));

                /* requested device found */
                return iter.operator *();
            }
        }
    }
    else
    {
        LOGD_DEBUG((demo, "AutoSmoketest::%s()  cannot find device, usbDevInfos is empty", __func__));
    }

    return nullptr;

}

void AutoSmoketest::createSignalTimer()
{
    struct sigevent sev;
    sev.sigev_notify = SIGEV_SIGNAL;
    sev.sigev_signo  = SIGINT;
    sev.sigev_value.sival_ptr = &timerid;

    if (timer_create(CLOCK_MONOTONIC, &sev, &timerid) == -1)
        fprintf(stderr,"AutoSmoketest::createSignalTimer timer_create failed ");
    else
        timerCreated = true;
}
void AutoSmoketest::startSignalTimer()
{
    struct itimerspec its;

    its.it_value.tv_sec     = stTimeout;
    its.it_value.tv_nsec    = 0;
    its.it_interval.tv_sec  = 0;
    its.it_interval.tv_nsec = 0;

    if (timer_settime(timerid, 0, &its, nullptr) == -1)
        fprintf(stderr, "AutoSmoketest::startSignalTimer timer_settime failed ");
}
void AutoSmoketest::deleteSignalTimer()
{
    if (true == timerCreated)
    {
        LOGD_DEBUG((demo, "AutoSmoketest::%s()  delete timer", __func__));
        timer_delete(timerid);
        timerCreated = false;
    }
}
std::string& AutoSmoketest::getErrorString(void)
{
   if (stError == NODEVICE)
   {
       errString = "NO DEVICE CONNECTED";
   }
   else if (stError == AOAPNOTSUPPORTED)
   {
       errString = "DEVICE DOES NOT SUPPORT AOAP MODE";
   }
   else if (stError == DEVICEALREADYINAOAP)
   {
       //todo:handle this error
       errString = "DEVICE ALREADY IN AOAP MODE";
   }
   else if (stError == DEVICENOTSWITCHED)
   {
       errString = "SWITCH SEND BUT DEVICE DOES NOT CHANGE MODE TO AOAP";
   }
   else if (stError == GSTINTERNAL)
   {
       errString = "notifyErrorCallback() Received. INTERNAL GSTREAMER ERROR" ;
   }
   else if (stError == NOERROR)
   {
       errString = "NO OCCURRED";
   }
   else if (stError == AAPSTARTED)
   {
       errString = "ANDROID AUTO PROJECTION STARTED";
   }
   else if (stError == VIDEOFRAMERENDERED)
   {
       errString = "VIDEO FRAME RENDERED";
   }
   else if (stError == DONE)
   {
       errString = "TEST EXECUTION DONE";
   }
   else
   {
       LOG_WARN((demo, "AutoSmoketest::getErrorString  Current stError %d is unknown", stError));
       errString = "UNKNOWN ERROR";
   }
   return errString;
}



} } // namespace adit { namespace aauto {
